<?php

namespace App\Repositries;

use App\Events\Chat\ChatEvent;
use App\Http\Traits\FireBaseTrait;

class ChatRepository
{
    use FireBaseTrait;
    public $user, $mainRepository, $conversation, $guard, $messages;

    public function __construct(MainRepository $mainRepository)
    {
        $this->user = auth()->guard('client')->check() ? auth()->guard('client')->user() : auth()->guard('designer')->user();
        $this->guard = auth()->guard('client')->check() ? 'client' : 'designer';
        $this->mainRepository = $mainRepository;
        $this->conversation = $this->user->conversations();
        $this->messages = $this->user->messages();
    }

    public function index()
    {

        return $this->mainRepository->index($this->conversation->whereHas('messages')->with(['messages' => function ($q) {
            return $q->latest()->take(1);
        }, 'designer', 'client']), request()->per_page ?? null);
    }


    public function startChatWith($request)
    {
        return $this->mainRepository->update($this->user, $this->user->id, ['chat_with' => $request->chat_with ?? null]);
    }

    public function startChat($data)
    {

        $conversation = $this->conversation->updateOrCreate([
            'designer_id' => $data['designer_id'],
            'client_id' => $this->user->id,
        ], [
            'designer_id' => $data['designer_id'],
            'client_id' => $this->user->id,
        ]);

        return [
            'status' => 'success',
            'code' => 200,
            'message' => __('lang.success'),
            'data' => $this->mainRepository->show($this->conversation, $conversation->id),
        ];
    }

    public function sendMessage($data)
    {
        $conversation = $this->mainRepository->show($this->conversation, $data['conversation_id']);
        $canChat = $conversation->can_chat;
        if (!$canChat) {
            return [
                'status' => 'failed',
                'code' => 422,
                'message' => __('lang.cant_chat')
            ];
        }

        $message = $this->mainRepository->create($this->messages, $data);

        if ($this->guard == 'client') {
            $receiver = $conversation->designer;
            $conversation->update([
                'designer_seen' => false,
            ]);
        } else {
            $receiver = $conversation->client;
            $conversation->update([
                'client_seen' => false,
            ]);
        }
        event(new ChatEvent($message));

        if ($receiver->chat_with != $this->user->id && $receiver->fcm_token) {
            $notification = [];
            $notification['title'] = $this->user->name;
            $notification['body'] = $message->message;
            $this->send($receiver->fcm_token, $notification);
        }

        return [
            'status' => 'success',
            'code' => 200,
            'message' => __('lang.success'),
            'data' => $message
        ];
    }

    public function show($id)
    {
        $per_page = request()->per_page ?? 10;
        $conversation =  $this->mainRepository->show($this->user->conversations(), $id)->load(['designer', 'client']);

        $conversation->update([
            $this->guard . '_seen' => true
        ]);
        $conversation['messages'] = $conversation->messages()->orderBy('created_at', 'desc')->take($per_page)->cursorPaginate($per_page);
        $conversation['messages']->setCollection(
            $conversation['messages']->getCollection()->reverse()->flatten()
        );
        return $conversation;
    }
}
