<?php

namespace App\Http\Controllers\API\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\API\Password\ConfirmOTPRequest;
use App\Http\Requests\API\Password\ForgetPasswordRequest;
use App\Http\Requests\API\Password\PasswordChangeRequest;
use App\Http\Requests\API\Password\ResetPasswordRequest;
use App\Http\Traits\ResponseTrait;
use App\Mail\SendOtp;
use App\Models\Client;
use App\Models\Otp;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Swift_TransportException;

class PasswordController extends Controller
{
    use ResponseTrait;
    public function changePassword(PasswordChangeRequest $request)
    {
        $user = auth('client')->user();
        if (!Hash::check($request->old_password, $user->password)) {
            return response()->json([
                'status'  => 'failed',
                'message' => __('auth.invalid_password')
            ], 422);
        }
        $user->update([
            'password' => $request->password
        ]);
        return $this->response(
            status: 'success',
            message: __('lang.success'),
            code: 200
        );
    }

    public function forgetPassword(ForgetPasswordRequest $request)
    {
        $client = Client::where('phone', $request->phone)->first();
        if ($client->status == '0') {
            return $this->response(
                status: 'failed',
                message: __('lang.inactive'),
                code: 422
            );
        }
        $otp    = rand(1000, 9999);
        $token  = Str::random(60);
        $client->otps()->create([
            'token'         => $token,
            'otp'           => $otp,
            'expired_at'    => now()->addMinutes(5),
        ]);

        try {
            Mail::to($client->email)->send(new SendOtp($otp));
        } catch (Swift_TransportException $exception) {
            Log::error($exception->getMessage());
        }

        return $this->response(
            data: $token,
            message: __('lang.otp_sent'),
        );
    }

    public function confirmOTP(ConfirmOTPRequest $request)
    {
        $otp        = $request->otp;
        $otpRecord  = Otp::where('otp', $otp)
            ->where('expired_at', '>', now())
            ->first();

        if (!$otpRecord) {
            return $this->response(
                status: 'failed',
                message: __('lang.invalid_otp'),
                code: 422
            );
        }

        $client       = $otpRecord->client;
        $clientTokent = Auth::guard('client')->login($client);

        return $this->response(
            data: [
                'token' => $clientTokent,
                'otp'   => $otpRecord,
            ],
        );
    }

    public function resetPassword(ResetPasswordRequest $request)
    {

        $otp = auth('client')->user()->otps()->where('token', $request['token'])
            ->first();

        if (!$otp) {
            return $this->response(
                status: 'failed',
                message: __('lang.invalid_token'),
                code: 422
            );
        }

        $otp->delete();
        auth('client')->user()->update([
            'password' => $request->new_password
        ]);
        return $this->response(
            message: __('lang.updated'),
            data: auth('client')->user(),
        );
    }
}
